@extends('layouts.app')

@section('main')
    <!-- Content Header (Page header) -->
    <section class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-6">
                    <h1><i class="fas fa-users"></i> Customer Management</h1>
                </div>
                <div class="col-sm-6">
                    <ol class="breadcrumb float-sm-right">
                        <li class="breadcrumb-item"><a href="{{ route('home') }}">Home</a></li>
                        <li class="breadcrumb-item active">Customers</li>
                    </ol>
                </div>
            </div>
        </div>
        
        @if (session('success'))
            <div class="alert alert-success alert-dismissible fade show">
                <button type="button" class="close" data-dismiss="alert">&times;</button>
                <i class="fas fa-check-circle"></i> {{ session('success') }}
            </div>
        @endif
        @if (session('error'))
            <div class="alert alert-danger alert-dismissible fade show">
                <button type="button" class="close" data-dismiss="alert">&times;</button>
                <i class="fas fa-exclamation-circle"></i> {{ session('error') }}
            </div>
        @endif
    </section>

    <!-- Main content -->
    <section class="content">
        <x-admin-table 
            title="All Customers"
            :create-route="route('customer.create')"
            create-text="Add Customer"
            :ajax-url="route('customers.index')"
            table-id="customers-table"
            :show-search="true"
            :show-date-filter="true"
            :show-export="true"
            :columns="[
                ['title' => 'SL', 'field' => 'sl', 'orderable' => false],
                ['title' => 'Name', 'field' => 'name'],
                ['title' => 'Email', 'field' => 'email'],
                ['title' => 'Mobile', 'field' => 'mobile'],
                ['title' => 'Address', 'field' => 'address'],
                ['title' => 'Total Due', 'field' => 'total_due'],
                ['title' => 'Total Paid', 'field' => 'total_paid'],
                ['title' => 'Total Credit', 'field' => 'total_credit'],
                ['title' => 'Advance', 'field' => 'advance'],
                ['title' => 'Cash Sales', 'field' => 'cash_sales'],
                ['title' => 'Credit Sales', 'field' => 'credit_sales'],
                ['title' => 'Exchange Sales', 'field' => 'exchange_sales'],
                ['title' => 'Joined Date', 'field' => 'created_at'],
                ['title' => 'Actions', 'field' => 'actions', 'orderable' => false, 'searchable' => false]
            ]"
        >
            <x-slot name="extraFilters">
                <div class="col-md-2">
                    <label for="due_filter" class="form-label">Due Status</label>
                    <select class="form-control" id="due_filter" name="due_filter">
                        <option value="">All Customers</option>
                        <option value="has_due">Has Due Amount</option>
                        <option value="no_due">No Due Amount</option>
                        <option value="high_due">High Due (>1000)</option>
                    </select>
                </div>
            </x-slot>
        </x-admin-table>
    </section>

    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
        // Customer action functions
        window.payDue = function(customerId, dueAmount) {
            if (dueAmount <= 0) {
                Swal.fire('Info', 'This customer has no due amount to pay.', 'info');
                return;
            }
            
            Swal.fire({
                title: 'Pay Due Amount',
                text: `Due Amount: BHD ${dueAmount.toFixed(2)}`,
                input: 'number',
                inputLabel: 'Amount to pay',
                inputValue: dueAmount.toFixed(2),
                inputAttributes: {
                    min: 0.01,
                    max: dueAmount,
                    step: 0.01
                },
                showCancelButton: true,
                confirmButtonText: 'Pay Now',
                confirmButtonColor: '#28a745',
                cancelButtonColor: '#6c757d',
                inputValidator: (value) => {
                    if (!value || value <= 0) {
                        return 'Please enter a valid amount!';
                    }
                    if (value > dueAmount) {
                        return 'Amount cannot exceed due amount!';
                    }
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    // Send AJAX request to pay due
                    $.ajax({
                        url: "{{ route('customer.payDue') }}",
                        method: 'POST',
                        data: {
                            customer_id: customerId,
                            amount: result.value,
                            type: 'due_payment',
                            notes: 'Due payment from customer management',
                            _token: $('meta[name="csrf-token"]').attr('content')
                        },
                        success: function(response) {
                            Swal.fire('Success!', response.message, 'success');
                            $('#customers-table').DataTable().ajax.reload();
                        },
                        error: function(xhr) {
                            let errorMsg = 'Failed to process payment.';
                            if (xhr.responseJSON && xhr.responseJSON.message) {
                                errorMsg = xhr.responseJSON.message;
                            }
                            Swal.fire('Error!', errorMsg, 'error');
                        }
                    });
                }
            });
        };

        window.addAdvance = function(customerId) {
            Swal.fire({
                title: 'Add Advance Payment',
                input: 'number',
                inputLabel: 'Advance Amount (BHD)',
                inputAttributes: {
                    min: 0.01,
                    step: 0.01
                },
                showCancelButton: true,
                confirmButtonText: 'Add Advance',
                confirmButtonColor: '#007bff',
                cancelButtonColor: '#6c757d',
                inputValidator: (value) => {
                    if (!value || value <= 0) {
                        return 'Please enter a valid amount!';
                    }
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    // Send AJAX request to add advance
                    $.ajax({
                        url: "{{ route('customer.payDue') }}",
                        method: 'POST',
                        data: {
                            customer_id: customerId,
                            amount: result.value,
                            type: 'advance',
                            notes: 'Advance payment from customer management',
                            _token: $('meta[name="csrf-token"]').attr('content')
                        },
                        success: function(response) {
                            Swal.fire('Success!', response.message, 'success');
                            $('#customers-table').DataTable().ajax.reload();
                        },
                        error: function(xhr) {
                            let errorMsg = 'Failed to add advance.';
                            if (xhr.responseJSON && xhr.responseJSON.message) {
                                errorMsg = xhr.responseJSON.message;
                            }
                            Swal.fire('Error!', errorMsg, 'error');
                        }
                    });
                }
            });
        };

        window.deleteCustomer = function(customerId) {
            Swal.fire({
                title: 'Are you sure?',
                text: "You won't be able to revert this!",
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#dc3545',
                cancelButtonColor: '#6c757d',
                confirmButtonText: 'Yes, delete it!'
            }).then((result) => {
                if (result.isConfirmed) {
                    // Send delete request
                    $.ajax({
                        url: `/customer/${customerId}`,
                        method: 'DELETE',
                        data: {
                            _token: $('meta[name="csrf-token"]').attr('content')
                        },
                        success: function(response) {
                            Swal.fire('Deleted!', 'Customer has been deleted.', 'success');
                            $('#customers-table').DataTable().ajax.reload();
                        },
                        error: function(xhr) {
                            let errorMsg = 'Failed to delete customer.';
                            if (xhr.responseJSON && xhr.responseJSON.message) {
                                errorMsg = xhr.responseJSON.message;
                            }
                            Swal.fire('Error!', errorMsg, 'error');
                        }
                    });
                }
            });
        };
    </script>
@endsection

@push('script')
<!-- jQuery (must be first) -->
<script src="{{ asset('assets/plugins/jquery/jquery.min.js') }}"></script>
<!-- DataTables core -->
<script src="{{ asset('assets/plugins/datatables/jquery.dataTables.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatables-bs4/js/dataTables.bootstrap4.min.js') }}"></script>
<!-- DataTables Buttons -->
<script src="{{ asset('assets/plugins/datatables-buttons/js/dataTables.buttons.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatables-buttons/js/buttons.bootstrap4.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatables-buttons/js/buttons.html5.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatables-buttons/js/buttons.print.min.js') }}"></script>
<script src="{{ asset('assets/plugins/datatables-buttons/js/buttons.colVis.min.js') }}"></script>

<script>
$(document).ready(function() {
    // Only initialize if not already initialized
    if (!$.fn.dataTable.isDataTable('#customers-table')) {
        window.customersTable = $('#customers-table').DataTable({
            dom: 'Bfrtip',
            buttons: ['copy', 'csv', 'excel', 'pdf', 'print'],
            // ...other options...
        });
    } else {
        window.customersTable = $('#customers-table').DataTable();
    }
});
// Use window.customersTable.ajax.reload() for updates
</script>
@endpush
